'use client';

import { useEffect, useRef } from 'react';
import { loadSEOSettings } from '@/lib/admin';

export default function SEOHead() {
  const refreshIntervalRef = useRef<NodeJS.Timeout | null>(null);

  const updateSEO = async () => {
    try {
      const seoSettings = await loadSEOSettings('global');
      if (seoSettings && typeof document !== 'undefined') {
        // Update title
        if (seoSettings.siteTitle) {
          document.title = seoSettings.siteTitle;
        }
        
        // Update meta tags
        const updateMetaTag = (name: string, content: string, attribute: string = 'name') => {
          let meta = document.querySelector(`meta[${attribute}="${name}"]`);
          if (!meta) {
            meta = document.createElement('meta');
            meta.setAttribute(attribute, name);
            document.head.appendChild(meta);
          }
          meta.setAttribute('content', content);
        };
        
        if (seoSettings.siteDescription) {
          updateMetaTag('description', seoSettings.siteDescription);
        }
        
        if (seoSettings.keywords) {
          updateMetaTag('keywords', seoSettings.keywords);
        }
        
        if (seoSettings.robots) {
          updateMetaTag('robots', seoSettings.robots);
        }
        
        if (seoSettings.author) {
          updateMetaTag('author', seoSettings.author);
        }
        
        // Update Open Graph tags
        if (seoSettings.ogTitle) {
          updateMetaTag('og:title', seoSettings.ogTitle, 'property');
        }
        
        if (seoSettings.ogDescription) {
          updateMetaTag('og:description', seoSettings.ogDescription, 'property');
        }
        
        if (seoSettings.ogImage) {
          updateMetaTag('og:image', seoSettings.ogImage, 'property');
        }
        
        if (seoSettings.canonicalUrl) {
          updateMetaTag('og:url', seoSettings.canonicalUrl, 'property');
          
          // Update canonical link
          let canonical = document.querySelector('link[rel="canonical"]');
          if (!canonical) {
            canonical = document.createElement('link');
            canonical.setAttribute('rel', 'canonical');
            document.head.appendChild(canonical);
          }
          canonical.setAttribute('href', seoSettings.canonicalUrl);
        }
        
        // Update Twitter tags
        if (seoSettings.twitterCard) {
          updateMetaTag('twitter:card', seoSettings.twitterCard);
        }
        
        if (seoSettings.twitterSite) {
          updateMetaTag('twitter:site', seoSettings.twitterSite);
        }
      }
    } catch (error) {
      console.error('Error updating SEO:', error);
    }
  };

  useEffect(() => {
    // Ensure viewport meta tag exists for mobile responsiveness
    if (typeof document !== 'undefined') {
      let viewport = document.querySelector('meta[name="viewport"]');
      if (!viewport) {
        viewport = document.createElement('meta');
        viewport.setAttribute('name', 'viewport');
        document.head.appendChild(viewport);
      }
      viewport.setAttribute('content', 'width=device-width, initial-scale=1, maximum-scale=5, user-scalable=yes');
    }

    // Initial update
    updateSEO();
    
    // Listen for SEO updates
    const handleSEOUpdate = () => {
      updateSEO();
    };
    
    window.addEventListener('seoSettingsUpdated', handleSEOUpdate);

    // Refresh data when window regains focus
    const handleFocus = () => {
      updateSEO();
    };

    // Refresh data when tab becomes visible
    const handleVisibilityChange = () => {
      if (!document.hidden) {
        updateSEO();
      }
    };

    window.addEventListener('focus', handleFocus);
    document.addEventListener('visibilitychange', handleVisibilityChange);

    // Periodic refresh every 15 seconds to catch updates from other devices/tabs
    refreshIntervalRef.current = setInterval(() => {
      updateSEO();
    }, 15000);
    
    return () => {
      window.removeEventListener('seoSettingsUpdated', handleSEOUpdate);
      window.removeEventListener('focus', handleFocus);
      document.removeEventListener('visibilitychange', handleVisibilityChange);
      if (refreshIntervalRef.current) {
        clearInterval(refreshIntervalRef.current);
      }
    };
  }, []);

  return null;
}

