'use client';

import { useState, useEffect } from 'react';
import { motion } from 'framer-motion';
import { useSearchParams } from 'next/navigation';
import { Send, CheckCircle, RefreshCw, Building2, User, Mail, Globe, MapPin, FileText, Upload, Shield, Package } from 'lucide-react';
import { useProducts } from '@/hooks/useProducts';
import { saveFormSubmission } from '@/lib/admin';

export default function EnquiryForm() {
  const products = useProducts();
  const searchParams = useSearchParams();
  const [formData, setFormData] = useState({
    companyName: '',
    contactPerson: '',
    email: '',
    country: '',
    customCountry: '',
    address: '',
    selectedProducts: [] as string[],
    requirements: '',
  });
  const [showResetConfirm, setShowResetConfirm] = useState(false);
  const [file, setFile] = useState<File | null>(null);
  const generateCaptcha = () => {
    const num1 = Math.floor(Math.random() * 10) + 1;
    const num2 = Math.floor(Math.random() * 10) + 1;
    return { num1, num2, answer: '' };
  };

  const [captcha, setCaptcha] = useState<{ num1: number; num2: number; answer: string } | null>(null);
  const [mounted, setMounted] = useState(false);

  useEffect(() => {
    setMounted(true);
    setCaptcha(generateCaptcha());
  }, []);

  // Auto-select product from URL parameter when products are loaded
  useEffect(() => {
    const productSlug = searchParams.get('product');
    if (productSlug && products.length > 0 && formData.selectedProducts.length === 0) {
      const product = products.find(p => p.slug === productSlug);
      if (product && product.id) {
        setFormData(prev => ({
          ...prev,
          selectedProducts: [product.id], // Auto-select the product
        }));
        console.log(`✅ Auto-selected product: ${product.name} (ID: ${product.id})`);
      }
    }
  }, [searchParams, products, formData.selectedProducts.length]);
  const [errors, setErrors] = useState<Record<string, string>>({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isSubmitted, setIsSubmitted] = useState(false);

  const validate = () => {
    const newErrors: Record<string, string> = {};

    if (!formData.companyName.trim()) {
      newErrors.companyName = 'Company name is required';
    } else if (formData.companyName.trim().length < 2) {
      newErrors.companyName = 'Company name must be at least 2 characters';
    }

    if (!formData.contactPerson.trim()) {
      newErrors.contactPerson = 'Contact person name is required';
    } else if (formData.contactPerson.trim().length < 2) {
      newErrors.contactPerson = 'Contact person name must be at least 2 characters';
    } else if (!/^[a-zA-Z\s]+$/.test(formData.contactPerson.trim())) {
      newErrors.contactPerson = 'Contact person name can only contain letters and spaces';
    }

    if (!formData.email.trim()) {
      newErrors.email = 'Email is required';
    } else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(formData.email)) {
      newErrors.email = 'Please enter a valid email address';
    }

    if (!formData.country.trim() && !formData.customCountry.trim()) {
      newErrors.country = 'Country is required';
    }

    if (!formData.address.trim()) {
      newErrors.address = 'Address is required';
    }

    if (!formData.requirements.trim()) {
      newErrors.requirements = 'Requirement details are required';
    }

    if (!captcha) {
      newErrors.captcha = 'Please wait for captcha to load';
    } else if (!captcha.answer.trim()) {
      newErrors.captcha = 'Please solve the captcha';
    } else if (parseInt(captcha.answer) !== captcha.num1 + captcha.num2) {
      newErrors.captcha = 'Incorrect answer. Please try again.';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData({ ...formData, [name]: value });
    if (errors[name]) {
      setErrors({ ...errors, [name]: '' });
    }
  };

  const handleProductChange = (productId: string) => {
    setFormData(prev => {
      const selectedProducts = prev.selectedProducts.includes(productId)
        ? prev.selectedProducts.filter(id => id !== productId)
        : [...prev.selectedProducts, productId];
      return { ...prev, selectedProducts };
    });
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      setFile(e.target.files[0]);
    }
  };

  const handleCaptchaChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (captcha) {
    setCaptcha({ ...captcha, answer: e.target.value });
    if (errors.captcha) {
      setErrors({ ...errors, captcha: '' });
      }
    }
  };

  const resetCaptcha = () => {
    const newCaptcha = generateCaptcha();
    setCaptcha({ ...newCaptcha, answer: captcha?.answer || '' });
  };

  const handleReset = () => {
    setFormData({
      companyName: '',
      contactPerson: '',
      email: '',
      country: '',
      customCountry: '',
      address: '',
      selectedProducts: [],
      requirements: '',
    });
    setFile(null);
    setCaptcha(generateCaptcha());
    setErrors({});
    setIsSubmitted(false);
    setShowResetConfirm(true);
    setTimeout(() => setShowResetConfirm(false), 3000);
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!validate()) {
      return;
    }

    setIsSubmitting(true);
    setErrors({}); // Clear previous errors

    try {
      // Get product names from IDs
      const productNames = formData.selectedProducts.map(productId => {
        const product = products.find(p => p.id === productId);
        return product ? product.name : productId;
      });

      // Convert file to base64 if exists
      let fileData = null;
      let fileType = null;
      if (file) {
        try {
          fileData = await new Promise<string>((resolve, reject) => {
            const reader = new FileReader();
            reader.onloadend = () => {
              resolve(reader.result as string);
            };
            reader.onerror = reject;
            reader.readAsDataURL(file);
          });
          fileType = file.type;
        } catch (error) {
          console.error('Error converting file to base64:', error);
        }
      }

      // Save form submission to admin dashboard (API)
      const result = await saveFormSubmission({
        source: 'enquiry',
        data: {
          companyName: formData.companyName,
          contactPerson: formData.contactPerson,
          email: formData.email,
          phone: '', // Enquiry form doesn't have phone, but we'll add it for consistency
          country: formData.country === 'other' ? formData.customCountry : formData.country,
          address: formData.address,
          selectedProducts: formData.selectedProducts, // Keep IDs for reference
          selectedProductNames: productNames, // Add names for display
          requirements: formData.requirements,
          fileName: file?.name || 'No file attached',
          fileData: fileData, // Base64 encoded file data
          fileType: fileType, // MIME type of the file
        },
      });
      
      if (result.success) {
        setIsSubmitted(true);
        // Reset after 5 seconds
        setTimeout(() => {
          handleReset();
        }, 5000);
      } else {
        // Show error to user
        setErrors({ submit: result.error || 'Failed to submit form. Please try again.' });
        console.error('Form submission failed:', result.error);
      }
    } catch (error: any) {
      console.error('Error submitting form:', error);
      setErrors({ submit: error.message || 'Failed to submit form. Please try again.' });
    } finally {
      // Always reset submitting state, even if there's an error
      setIsSubmitting(false);
    }
  };

  return (
    <div className="pt-44 sm:pt-48 md:pt-56">
      {/* Hero Section */}
      <section className="py-10 sm:py-12 md:py-16 bg-gradient-to-br bg-gradient-hero border-b border-slate-200">
        <motion.div
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.8 }}
          className="container mx-auto px-4 sm:px-6 md:px-8 text-center max-w-4xl"
        >
          <h1 className="text-3xl sm:text-4xl md:text-5xl font-bold mb-3 sm:mb-4 text-slate-900 px-2 sm:px-0">Enquiry Form</h1>
          <p className="text-base sm:text-lg md:text-xl text-slate-700 px-2 sm:px-0">
            Submit your requirements and get a customized quote
          </p>
        </motion.div>
      </section>

      {/* Form Section */}
      <section className="py-10 sm:py-12 md:py-16 bg-beige-50">
        <div className="container mx-auto px-4 sm:px-6 md:px-8 max-w-3xl">
          <motion.div
            initial={{ opacity: 0, y: 30 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.8, delay: 0.2 }}
            className="bg-white rounded-2xl shadow-xl p-5 sm:p-6 md:p-10 border border-slate-200"
          >
            {isSubmitted && (
              <motion.div
                initial={{ opacity: 0, scale: 0.95 }}
                animate={{ opacity: 1, scale: 1 }}
                className="mb-6 p-4 bg-gradient-to-r from-green-50 to-emerald-50 border-2 border-green-400 text-green-800 rounded-xl flex items-center space-x-3 shadow-lg"
              >
                <CheckCircle size={24} className="text-green-600 flex-shrink-0" />
                <div>
                  <p className="font-bold text-green-800">Success!</p>
                  <p className="text-sm text-green-700">Enquiry submitted successfully! We'll get back to you soon.</p>
                </div>
              </motion.div>
            )}

            {showResetConfirm && (
              <motion.div
                initial={{ opacity: 0, scale: 0.95 }}
                animate={{ opacity: 1, scale: 1 }}
                exit={{ opacity: 0, scale: 0.95 }}
                className="mb-6 p-4 bg-gradient-to-r from-blue-50 to-sky-50 border-2 border-blue-400 text-blue-800 rounded-xl flex items-center space-x-3 shadow-lg"
              >
                <CheckCircle size={24} className="text-blue-600 flex-shrink-0" />
                <div>
                  <p className="font-bold text-blue-800">Reset Complete!</p>
                  <p className="text-sm text-blue-700">Form has been reset successfully. You can now fill it again.</p>
                </div>
              </motion.div>
            )}

            <form onSubmit={handleSubmit} className="space-y-5">
              {/* Company Name */}
              <div>
                <label htmlFor="companyName" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                  <Building2 size={18} className="text-primary-600" />
                  Name of Company <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  id="companyName"
                  name="companyName"
                  value={formData.companyName}
                  onChange={handleChange}
                  className={`w-full px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all ${
                    errors.companyName ? 'border-red-500' : 'border-slate-300'
                  }`}
                  placeholder="Enter company name"
                />
                {errors.companyName && (
                  <p className="mt-1 text-sm text-red-500">{errors.companyName}</p>
                )}
              </div>

              {/* Contact Person */}
              <div>
                <label htmlFor="contactPerson" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                  <User size={18} className="text-primary-600" />
                  Name of Contact Person <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  id="contactPerson"
                  name="contactPerson"
                  value={formData.contactPerson}
                  onChange={handleChange}
                  className={`w-full px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all ${
                    errors.contactPerson ? 'border-red-500' : 'border-slate-300'
                  }`}
                  placeholder="Enter contact person name"
                />
                {errors.contactPerson && (
                  <p className="mt-1 text-sm text-red-500">{errors.contactPerson}</p>
                )}
              </div>

              {/* Email */}
              <div>
                <label htmlFor="email" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                  <Mail size={18} className="text-primary-600" />
                  Email <span className="text-red-500">*</span>
                </label>
                <input
                  type="email"
                  id="email"
                  name="email"
                  value={formData.email}
                  onChange={handleChange}
                  className={`w-full px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all ${
                    errors.email ? 'border-red-500' : 'border-slate-300'
                  }`}
                  placeholder="your.email@example.com"
                />
                {errors.email && (
                  <p className="mt-1 text-sm text-red-500">{errors.email}</p>
                )}
              </div>

              {/* Country */}
              <div>
                <label htmlFor="country" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                  <Globe size={18} className="text-primary-600" />
                  Country <span className="text-red-500">*</span>
                </label>
                <select
                  id="country"
                  name="country"
                  value={formData.country}
                  onChange={handleChange}
                  className={`w-full px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all bg-white ${
                    errors.country ? 'border-red-500' : 'border-slate-300'
                  }`}
                >
                  <option value="">Select a country</option>
                  <option value="India">India</option>
                  <option value="United States">United States</option>
                  <option value="United Kingdom">United Kingdom</option>
                  <option value="Canada">Canada</option>
                  <option value="Australia">Australia</option>
                  <option value="Germany">Germany</option>
                  <option value="France">France</option>
                  <option value="Japan">Japan</option>
                  <option value="China">China</option>
                  <option value="Singapore">Singapore</option>
                  <option value="UAE">UAE</option>
                  <option value="Saudi Arabia">Saudi Arabia</option>
                  <option value="Other">Other (specify below)</option>
                </select>
                {formData.country === 'Other' && (
                  <input
                    type="text"
                    id="customCountry"
                    name="customCountry"
                    value={formData.customCountry}
                    onChange={handleChange}
                    className={`w-full mt-2 px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all ${
                      errors.country ? 'border-red-500' : 'border-slate-300'
                    }`}
                    placeholder="Enter your country"
                />
                )}
                {errors.country && (
                  <p className="mt-1 text-sm text-red-500 font-medium">{errors.country}</p>
                )}
              </div>

              {/* Address */}
              <div>
                <label htmlFor="address" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                  <MapPin size={18} className="text-primary-600" />
                  Address/City/Location <span className="text-red-500">*</span>
                </label>
                <textarea
                  id="address"
                  name="address"
                  rows={3}
                  value={formData.address}
                  onChange={handleChange}
                  className={`w-full px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all resize-none ${
                    errors.address ? 'border-red-500' : 'border-slate-300'
                  }`}
                  placeholder="Enter address, city, or location"
                />
                {errors.address && (
                  <p className="mt-1 text-sm text-red-500">{errors.address}</p>
                )}
              </div>

              {/* Product Selection */}
              <div>
                <label className="flex items-center gap-2 text-slate-900 font-semibold mb-3">
                  <Package size={18} className="text-primary-600" />
                  Select Products (Optional)
                </label>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-3 max-h-64 overflow-y-auto p-3 border border-slate-300 rounded-xl bg-slate-50">
                  {products.map((product) => (
                    <label
                      key={product.id}
                      className="flex items-center space-x-2 p-2 rounded-lg hover:bg-white cursor-pointer transition-colors"
                    >
                      <input
                        type="checkbox"
                        checked={formData.selectedProducts.includes(product.id)}
                        onChange={() => handleProductChange(product.id)}
                        className="w-4 h-4 text-primary-600 border-slate-300 rounded focus:ring-primary-500"
                      />
                      <span className="text-sm text-slate-700 font-medium">{product.name}</span>
                    </label>
                  ))}
                </div>
                {formData.selectedProducts.length > 0 && (
                  <p className="mt-2 text-sm text-slate-600">
                    {formData.selectedProducts.length} product(s) selected
                  </p>
                )}
              </div>

              {/* Requirements */}
              <div>
                <label htmlFor="requirements" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                  <FileText size={18} className="text-primary-600" />
                  Requirement Detail <span className="text-red-500">*</span>
                </label>
                <textarea
                  id="requirements"
                  name="requirements"
                  rows={4}
                  value={formData.requirements}
                  onChange={handleChange}
                  className={`w-full px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all resize-none ${
                    errors.requirements ? 'border-red-500' : 'border-slate-300'
                  }`}
                  placeholder="Describe your packaging requirements in detail..."
                />
                {errors.requirements && (
                  <p className="mt-1 text-sm text-red-500">{errors.requirements}</p>
                )}
              </div>

              {/* File Upload */}
              <div>
                <label htmlFor="file" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                  <Upload size={18} className="text-primary-600" />
                  Choose File (Optional)
                </label>
                <div className="relative">
                <input
                  type="file"
                  id="file"
                  name="file"
                  onChange={handleFileChange}
                    className="w-full px-4 py-2.5 border border-slate-300 rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all file:mr-4 file:py-1 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-primary-50 file:text-primary-700 hover:file:bg-primary-100"
                  accept=".pdf,.doc,.docx,.jpg,.jpeg,.png"
                />
                </div>
                {file && (
                  <p className="mt-2 text-sm text-slate-600 flex items-center gap-2">
                    <Upload size={14} /> Selected: {file.name}
                  </p>
                )}
              </div>

              {/* Captcha */}
              {mounted && captcha && (
              <div>
                  <label htmlFor="captcha" className="flex items-center gap-2 text-slate-900 font-semibold mb-2">
                    <Shield size={18} className="text-primary-600" />
                  Captcha Verification <span className="text-red-500">*</span>
                </label>
                  <div className="flex items-center gap-3">
                    <div className="flex items-center space-x-2 px-4 py-2.5 bg-gradient-to-r from-primary-50 to-sky-50 border-2 border-primary-200 rounded-xl">
                      <span className="text-xl font-bold text-slate-900">
                      {captcha.num1} + {captcha.num2} = ?
                    </span>
                  </div>
                  <input
                    type="number"
                    id="captcha"
                    name="captcha"
                    value={captcha.answer}
                    onChange={handleCaptchaChange}
                      className={`flex-1 px-4 py-2.5 border rounded-xl focus:ring-2 focus:ring-primary-500 focus:border-primary-500 outline-none transition-all ${
                      errors.captcha ? 'border-red-500' : 'border-slate-300'
                    }`}
                      placeholder="Answer"
                  />
                  <button
                    type="button"
                    onClick={resetCaptcha}
                      className="p-2.5 bg-slate-100 hover:bg-slate-200 rounded-xl transition-colors border border-slate-300"
                    aria-label="Refresh captcha"
                  >
                      <RefreshCw size={18} className="text-slate-700" />
                  </button>
                </div>
                {errors.captcha && (
                    <p className="mt-1 text-sm text-red-500 font-medium">{errors.captcha}</p>
                )}
              </div>
              )}

              {errors.submit && (
                <div className="p-4 bg-red-50 border-2 border-red-400 text-red-800 rounded-xl">
                  <p className="font-bold text-red-800">Error</p>
                  <p className="text-sm text-red-700">{errors.submit}</p>
                </div>
              )}

              {/* Buttons */}
              <div className="flex flex-col sm:flex-row gap-3 pt-2">
                <button
                  type="submit"
                  disabled={isSubmitting}
                  className="flex-1 btn-primary flex items-center justify-center space-x-2 disabled:opacity-50 disabled:cursor-not-allowed py-3"
                >
                  {isSubmitting ? (
                    <>
                      <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin" />
                      <span>Submitting...</span>
                    </>
                  ) : (
                    <>
                      <Send size={20} />
                      <span>Submit Enquiry</span>
                    </>
                  )}
                </button>
                <button
                  type="button"
                  onClick={handleReset}
                  className="flex-1 btn-secondary py-3"
                >
                  Reset Form
                </button>
              </div>
            </form>
          </motion.div>
        </div>
      </section>
    </div>
  );
}

